// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


package com.vasco.digipass.sdk.utils.qrcodescanner.sample;

import android.Manifest;
import android.app.Activity;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.widget.LinearLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import androidx.appcompat.app.AlertDialog;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.core.graphics.Insets;
import androidx.core.view.OnApplyWindowInsetsListener;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;
import androidx.core.view.WindowInsetsControllerCompat;

import com.vasco.digipass.sdk.utils.qrcodescanner.QRCodeScannerSDKConstants;
import com.vasco.digipass.sdk.utils.qrcodescanner.QRCodeScannerSDKErrorCodes;
import com.vasco.digipass.sdk.utils.qrcodescanner.QRCodeScannerSDKException;

import java.util.Objects;

public class QrCodeScannerSampleActivity extends Activity
        implements ActivityCompat.OnRequestPermissionsResultCallback {

    private static final String TAG = QrCodeScannerSampleActivity.class.getSimpleName();
    private String errorQRCode = "";

    /** Constant for retrieving the permission result. */
    static final int ACCESS_CAMERA_REQUEST_CODE = 1;

    /** Request code used for Camera's activity results. */
    private static final int CAMERA_ACTVITY_REQUEST_CODE = 2;

    /** The views displaying the result of the QR code scan. */
    private TextView resultTextView;

    private TextView resultTextType;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_sample_main);
        errorQRCode = getString(R.string.error_qr_Code);

        // Instantiate the Text Views
        resultTextView = findViewById(R.id.text);
        resultTextType = findViewById(R.id.text_code_type);

        if (ContextCompat.checkSelfPermission(this, Manifest.permission.CAMERA)
                != PackageManager.PERMISSION_GRANTED) {
            requestPermissions();
        }

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            View rootView = findViewById(R.id.rootView);
            setInsets(getWindow(), rootView);
        }
    }

    @RequiresApi(api = Build.VERSION_CODES.Q)
    public void setInsets(Window window, View rootView) {
        WindowInsetsControllerCompat controller =
                new WindowInsetsControllerCompat(window, window.getDecorView());
        controller.setAppearanceLightStatusBars(true);
        ViewCompat.setOnApplyWindowInsetsListener(rootView, new OnApplyWindowInsetsListener() {
            @NonNull
            @Override
            public WindowInsetsCompat onApplyWindowInsets(@NonNull View v, @NonNull WindowInsetsCompat insets) {
                int systemBars = WindowInsetsCompat.Type.systemBars();
                Insets systemBarInsets = insets.getInsets(systemBars);

                ViewGroup.MarginLayoutParams layoutParams = (ViewGroup.MarginLayoutParams) v.getLayoutParams();
                layoutParams.topMargin = systemBarInsets.top;
                layoutParams.bottomMargin = systemBarInsets.bottom;
                layoutParams.leftMargin = systemBarInsets.left;
                layoutParams.rightMargin = systemBarInsets.right;

                v.setLayoutParams(layoutParams);
                return WindowInsetsCompat.CONSUMED;
            }
        });
    }

    /** Decodes an image using custom cameraX implementation */
    public void launchSystemCamera(View v) {
        if (ContextCompat.checkSelfPermission(this, Manifest.permission.CAMERA)
                != PackageManager.PERMISSION_GRANTED) {
            requestPermissions();
        } else {
            Intent intent = new Intent(this, SystemCameraActivity.class);
            this.startActivityForResult(intent, CAMERA_ACTVITY_REQUEST_CODE);
        }
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (resultCode == RESULT_OK) { // The result is returned as an extra
            int scannedImageFormat =
                    data.getIntExtra(QRCodeScannerSDKConstants.OUTPUT_CODE_TYPE, 0);
            String scannedImageData = data.getStringExtra(QRCodeScannerSDKConstants.OUTPUT_RESULT);

            String format =
                    scannedImageFormat == QRCodeScannerSDKConstants.CRONTO_CODE
                            ? "Cronto Sign"
                            : "QR Code";
            Log.d(TAG, "Scanned image data = " + scannedImageData);
            Log.d(TAG, "Scanned image format = " + format);

            resultTextView.setText(scannedImageData);
            resultTextType.setText(getResources().getString(R.string.scanned_data_info, format));
        } else if (resultCode == RESULT_CANCELED) {
            resultTextView.setText(getResources().getString(R.string.scan_cancelled));
            resultTextType.setText("");
        } else if (resultCode == QRCodeScannerSDKConstants.RESULT_ERROR) { // Get returned exception
            QRCodeScannerSDKException exception =
                    (QRCodeScannerSDKException)
                            data.getSerializableExtra(QRCodeScannerSDKConstants.OUTPUT_EXCEPTION);
            onExceptionThrown(exception);
        }
    }

    /**
     * Closes the activity if an exception has been raised
     *
     * @param e exception thrown
     */
    private void onExceptionThrown(Throwable e) {
        String errorMessage;
        if (e instanceof QRCodeScannerSDKException) {
            int errorCode = ((QRCodeScannerSDKException) e).getErrorCode();
            Log.e(TAG, "onExceptionThrown: ", e);
            errorMessage = switch (errorCode) {
                case QRCodeScannerSDKErrorCodes.CAMERA_NOT_AVAILABLE ->
                        "Camera is not available on this device";
                case QRCodeScannerSDKErrorCodes.PERMISSION_DENIED ->
                        "The permission for accessing the camera is not set";
                case QRCodeScannerSDKErrorCodes.NATIVE_LIBRARY_NOT_LOADED ->
                        "The native library cannot be loaded";
                case QRCodeScannerSDKErrorCodes.INVALID_CODE_TYPE ->
                        "The given code type is invalid";
                case QRCodeScannerSDKErrorCodes.INVALID_IMAGE -> "The scanned image is invalid";
                case QRCodeScannerSDKErrorCodes.INTERNAL_ERROR ->
                        errorQRCode + Objects.requireNonNull(e.getCause()).getMessage();
                default -> errorQRCode + errorCode;
            };
        } else {
            errorMessage = errorQRCode + Objects.requireNonNull(e.getCause()).getMessage();
        }
        Log.e(TAG, errorMessage);
        resultTextView.setText(errorMessage);
        resultTextType.setText("");
    }

    private void requestPermissions() {
        if (ActivityCompat.shouldShowRequestPermissionRationale(this, Manifest.permission.CAMERA)) {
            // Permission was already denied.
            // Display a clear explanation about why this permission is important for the app to
            // work
            // Then, request the permission again
            AlertDialog.Builder builder = new AlertDialog.Builder(this);
            builder.setTitle("Permission request");
            builder.setMessage("The requested permission is used to scan codes.");
            builder.setOnCancelListener(this::onCancel);
            builder.create().show();
        } else {
            // Request permission for the CAMERA permission
            ActivityCompat.requestPermissions(
                    this, new String[] {Manifest.permission.CAMERA}, ACCESS_CAMERA_REQUEST_CODE);
        }
    }

    @Override
    public void onRequestPermissionsResult(
            int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        if (requestCode
                == ACCESS_CAMERA_REQUEST_CODE) { // If request is cancelled, the result arrays are
            // empty.
            if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                // Permission was granted.
            } else {
                // Permission was denied.
                // In our case, we could try to start the camera anyway since is would
                // return a "permission denied" error code, but we can shortcut that process
                // by directly changing the result textView
                resultTextView.setVisibility(View.VISIBLE);
                resultTextView.setText(getResources().getString(R.string.permission_not_granted));
            }
        }
    }

    @Override
    public void onBackPressed() {
        super.onBackPressed();
        finish();
    }

    private void onCancel(DialogInterface dialog) {
        // Request permission for the CAMERA permission
        ActivityCompat.requestPermissions(
                QrCodeScannerSampleActivity.this,
                new String[] {Manifest.permission.CAMERA},
                ACCESS_CAMERA_REQUEST_CODE);
    }
}
